<?php
// camera_settings.php
// صفحهٔ نهایی مدیریت تنظیمات دوربین‌ها با فیلتر، مرتب‌سازی، ذخیرهٔ تکی و اعمال گروهی

session_start();
if (!isset($_SESSION['loggedin'])) {
    header('Location: login.php');
    exit();
}

require_once __DIR__ . '/db.php';
require_once __DIR__ . '/camera_settings_model.php';

$pdo = getPDO();

/* -----------------------------
   ابزارک‌های کمکی (Flash/PRG/Query)
------------------------------ */
function cs_h(string $v): string { return htmlspecialchars($v, ENT_QUOTES, 'UTF-8'); }

function cs_flash_add(string $key, string $type, string $message): void {
    $_SESSION['flash'][$key] = ['type' => $type, 'message' => $message];
}
function cs_flash_get(string $key): ?array {
    if (!isset($_SESSION['flash'][$key])) return null;
    $msg = $_SESSION['flash'][$key];
    unset($_SESSION['flash'][$key]);
    return $msg;
}
function cs_build_query(array $state): string {
    // مقادیر خالی حذف شوند
    $filtered = [];
    foreach ($state as $k => $v) {
        if ($v === null || $v === '') continue;
        $filtered[$k] = $v;
    }
    $q = http_build_query($filtered);
    return $q ? ('?' . $q) : '';
}
function cs_redirect_with_state(array $state, ?string $hash = null): void {
    $url = 'camera_settings.php' . cs_build_query($state);
    if ($hash) $url .= '#' . $hash;
    header("Location: $url");
    exit();
}
function cs_format_dt(?string $dt): string {
    if (!$dt) return '-';
    // اگر رشتهٔ DATETIME از DB باشد
    $ts = strtotime($dt);
    if ($ts === false) return cs_h($dt);
    return date('Y/m/d H:i', $ts);
}

/* -----------------------------
   ورودی‌های فیلتر و مرتب‌سازی
------------------------------ */
$q            = trim($_GET['q'] ?? '');
$frameFilter  = strtoupper(trim($_GET['frame'] ?? '')); // کلیدهای ALLOWED_FRAMESIZES
$captureParam = $_GET['capture'] ?? '';                 // '1' | '0' | ''
$sortKey      = $_GET['sort'] ?? 'device_asc';          // پیش‌فرض: device_id صعودی (مطابق نیاز)

$sortOptions = [
    'device_asc'  => 'ID صعودی',
    'device_desc' => 'ID نزولی',
    'name_asc'    => 'نام صعودی',
    'name_desc'   => 'نام نزولی',
    'updated_desc'=> 'آخرین بروزرسانی (جدیدترین)',
    'updated_asc' => 'آخرین بروزرسانی (قدیمی‌ترین)',
];
if (!array_key_exists($sortKey, $sortOptions)) {
    $sortKey = 'device_asc';
}

$stateQuery = [
    'q'       => $q !== '' ? $q : null,
    'frame'   => $frameFilter !== '' ? $frameFilter : null,
    'capture' => ($captureParam === '1' || $captureParam === '0') ? $captureParam : null,
    'sort'    => $sortKey !== 'device_asc' ? $sortKey : null, // فقط اگر با پیش‌فرض فرق دارد در URL بماند
];

/* -----------------------------
   هندلر ذخیرهٔ تکی کارت (Inline)
------------------------------ */
if ($_SERVER['REQUEST_METHOD'] === 'POST' && ($_POST['action'] ?? '') === 'save_single') {
    $deviceId = trim($_POST['device_id'] ?? '');
    if ($deviceId === '') {
        cs_flash_add('global', 'error', 'شناسهٔ دستگاه ارسال نشده است.');
        cs_redirect_with_state($stateQuery);
    }

    $payload = [
        'interval_minutes' => $_POST['interval_minutes'] ?? null,
        'quality'          => $_POST['quality'] ?? null,
        'frame_size'       => $_POST['frame_size'] ?? null,
        // چک‌باکس اگر تیک نخورده باشد در POST نمی‌آید؛ بنابراین به‌صورت صریح مقداردهی می‌کنیم
        'capture_enabled'  => isset($_POST['capture_enabled']) ? '1' : '0',
    ];

    try {
        $result = saveCameraSettings($pdo, $deviceId, $payload);
        if ($result['ok']) {
            cs_flash_add("cam-$deviceId", 'success', 'تنظیمات با موفقیت ذخیره شد.');
        } else {
            // ساخت پیام خطا از فهرست فیلدها
            $aliases = [
                'interval_minutes' => 'بازهٔ زمانی',
                'quality'          => 'کیفیت',
                'frame_size'       => 'اندازهٔ فریم',
                'capture_enabled'  => 'کپچر',
            ];
            $parts = [];
            foreach ($result['errors'] as $f => $m) {
                $label = $aliases[$f] ?? $f;
                $parts[] = "[$label] $m";
            }
            $msg = $parts ? implode(' ؛ ', $parts) : 'خطای نامشخص در ذخیرهٔ تنظیمات.';
            cs_flash_add("cam-$deviceId", 'error', $msg);
        }
    } catch (Throwable $e) {
        cs_flash_add("cam-$deviceId", 'error', 'خطای داخلی هنگام ذخیره تنظیمات.');
    }

    cs_redirect_with_state($stateQuery, 'cam-' . rawurlencode($deviceId));
}

/* -----------------------------
   هندلر اعمال گروهی (Bulk)
------------------------------ */
if ($_SERVER['REQUEST_METHOD'] === 'POST' && ($_POST['action'] ?? '') === 'bulk_apply') {
    // دستگاه‌های انتخاب‌شده از طریق JS در یک رشتهٔ comma-separated ارسال می‌شوند
    $bulkDevicesRaw = trim($_POST['bulk_devices'] ?? '');
    $ids = array_values(array_filter(array_map('trim', explode(',', $bulkDevicesRaw)), fn($v) => $v !== ''));

    if (empty($ids)) {
        cs_flash_add('global', 'error', 'هیچ دوربینی انتخاب نشده است.');
        cs_redirect_with_state($stateQuery);
    }

    // تعیین اینکه کدام فیلدها باید اعمال شوند
    $applyInterval = isset($_POST['apply_interval']);
    $applyQuality  = isset($_POST['apply_quality']);
    $applyFrame    = isset($_POST['apply_frame_size']);
    $applyCapture  = isset($_POST['apply_capture']);

    // اگر هیچ فیلدی برای اعمال تیک نخورده باشد
    if (!$applyInterval && !$applyQuality && !$applyFrame && !$applyCapture) {
        cs_flash_add('global', 'error', 'هیچ فیلدی برای اعمال گروهی انتخاب نشده است.');
        cs_redirect_with_state($stateQuery);
    }

    // مقادیر ورودی Bulk (در صورت انتخاب)
    $bulkInterval = $_POST['bulk_interval_minutes'] ?? null;
    $bulkQuality  = $_POST['bulk_quality'] ?? null;
    $bulkFrame    = $_POST['bulk_frame_size'] ?? null;
    $bulkCapture  = $_POST['bulk_capture_enabled'] ?? null; // '1' | '0'

    $okCount = 0;
    $failCount = 0;

    foreach ($ids as $deviceId) {
        try {
            // برای پاس‌دادن کامل به validate، باید مقادیر فعلی را بگیریم
            $current = getCameraSettings($pdo, $deviceId);
            if ($current === null) {
                $failCount++;
                cs_flash_add("cam-$deviceId", 'error', 'دستگاه یافت نشد.');
                continue;
            }

            $payload = [
                'interval_minutes' => $applyInterval ? $bulkInterval : $current['interval_minutes'],
                'quality'          => $applyQuality  ? $bulkQuality  : $current['quality'],
                'frame_size'       => $applyFrame    ? $bulkFrame    : $current['frame_size'],
                'capture_enabled'  => $applyCapture  ? ($bulkCapture ?? ($current['capture_enabled'] ? '1' : '0'))
                                                     : ($current['capture_enabled'] ? '1' : '0'),
            ];

            $res = saveCameraSettings($pdo, $deviceId, $payload);
            if ($res['ok']) {
                $okCount++;
                cs_flash_add("cam-$deviceId", 'success', 'بروزرسانی گروهی با موفقیت اعمال شد.');
            } else {
                $failCount++;
                $aliases = [
                    'interval_minutes' => 'بازهٔ زمانی',
                    'quality'          => 'کیفیت',
                    'frame_size'       => 'اندازهٔ فریم',
                    'capture_enabled'  => 'کپچر',
                ];
                $parts = [];
                foreach ($res['errors'] as $f => $m) {
                    $parts[] = '[' . ($aliases[$f] ?? $f) . "] $m";
                }
                cs_flash_add("cam-$deviceId", 'error', ($parts ? implode(' ؛ ', $parts) : 'خطا در اعمال گروهی.'));
            }
        } catch (Throwable $e) {
            $failCount++;
            cs_flash_add("cam-$deviceId", 'error', 'خطای داخلی هنگام اعمال گروهی.');
        }
    }

    if ($okCount > 0 && $failCount === 0) {
        cs_flash_add('global', 'success', "گروه‌بندی: $okCount دستگاه با موفقیت بروزرسانی شد.");
    } elseif ($okCount > 0 && $failCount > 0) {
        cs_flash_add('global', 'error', "بخشی از عملیات انجام شد: موفق $okCount / ناموفق $failCount");
    } else {
        cs_flash_add('global', 'error', 'اعمال گروهی ناموفق بود.');
    }

    cs_redirect_with_state($stateQuery);
}

/* -----------------------------
   واکشی دوربین‌ها و اعمال فیلتر/مرتب‌سازی
------------------------------ */
try {
    $allCameras = getAllCameras($pdo);
} catch (Throwable $e) {
    $allCameras = [];
}

// نگاشت به ساختار نمایشی
$items = [];
foreach ($allCameras as $row) {
    $devId = (string)$row['device_id'];
    $name  = isset($row['name']) && $row['name'] !== '' ? (string)$row['name'] : null;

    $s = is_array($row['settings']) ? $row['settings'] : [];
    // در مدل، settings قبلاً merge و normalize شده است
    $items[] = [
        'device_id'        => $devId,
        'display_name'     => $name ?: $devId,
        'name'             => $name,
        'interval_minutes' => (int)($s['interval_minutes'] ?? 60),
        'quality'          => (int)($s['quality'] ?? 10),
        'frame_size'       => (string)($s['frame_size'] ?? 'SVGA'),
        'capture_enabled'  => !empty($s['capture_enabled']),
        'created_at'       => $row['created_at'] ?? null,
        'updated_at'       => $row['updated_at'] ?? null,
    ];
}

// فیلتر متن
if ($q !== '') {
    $qLower = mb_strtolower($q, 'UTF-8');
    $items = array_values(array_filter($items, function ($it) use ($qLower) {
        $hay1 = mb_strtolower($it['device_id'], 'UTF-8');
        $hay2 = mb_strtolower($it['display_name'], 'UTF-8');
        return (mb_strpos($hay1, $qLower, 0, 'UTF-8') !== false) ||
               (mb_strpos($hay2, $qLower, 0, 'UTF-8') !== false);
    }));
}

// فیلتر اندازهٔ فریم
if ($frameFilter !== '' && isset(ALLOWED_FRAMESIZES[$frameFilter])) {
    $items = array_values(array_filter($items, fn($it) => strtoupper($it['frame_size']) === $frameFilter));
}

// فیلتر وضعیت کپچر
if ($captureParam === '1' || $captureParam === '0') {
    $want = $captureParam === '1';
    $items = array_values(array_filter($items, fn($it) => (bool)$it['capture_enabled'] === $want));
}

// مرتب‌سازی
usort($items, function ($a, $b) use ($sortKey) {
    switch ($sortKey) {
        case 'device_desc':
            return strnatcasecmp($b['device_id'], $a['device_id']);
        case 'name_asc':
            return strnatcasecmp($a['display_name'], $b['display_name']) ?: strnatcasecmp($a['device_id'], $b['device_id']);
        case 'name_desc':
            return strnatcasecmp($b['display_name'], $a['display_name']) ?: strnatcasecmp($a['device_id'], $b['device_id']);
        case 'updated_desc':
            return strcmp($b['updated_at'] ?? '', $a['updated_at'] ?? '');
        case 'updated_asc':
            return strcmp($a['updated_at'] ?? '', $b['updated_at'] ?? '');
        case 'device_asc':
        default:
            return strnatcasecmp($a['device_id'], $b['device_id']);
    }
});

$resultCount = count($items);
?>
<!DOCTYPE html>
<html lang="fa" dir="rtl">
<head>
<meta charset="UTF-8">
<meta name="viewport" content="width=device-width, initial-scale=1, maximum-scale=1">
<title>تنظیمات دوربین‌ها</title>
<link rel="stylesheet" href="style.css">
<style>
/* حداقل استایل‌های تکمیلی برای کارت‌ها و پنل اعمال گروهی */
.settings-page { padding: 16px; }
.panel-heading { display:flex; align-items:center; justify-content:space-between; gap:12px; margin-bottom:12px; }
.count-chip { background: rgba(255,255,255,0.08); border:1px solid rgba(255,255,255,0.15); border-radius:999px; padding:6px 10px; font-size:.85rem; }
.filters { display:grid; grid-template-columns: repeat(auto-fit,minmax(180px, 1fr)); gap:10px; margin-bottom:12px; }
.filters .filter-item label { display:block; font-size:.86rem; margin-bottom:6px; color:#cfd6ea; }
.filters .filter-item input[type=text],
.filters .filter-item select { width:100%; padding:8px 10px; border-radius:10px; border:1px solid rgba(255,255,255,0.15); background:rgba(255,255,255,0.06); color:#fff; }

.apply-board { display:flex; flex-direction:column; gap:10px; padding:12px; border:1px solid rgba(255,255,255,0.12); border-radius:14px; background:rgba(9,20,40,0.9); margin-bottom:12px; }
.apply-row { display:grid; grid-template-columns: repeat(auto-fit,minmax(160px, 1fr)); gap:10px; }
.apply-row .field { display:flex; gap:6px; align-items:center; }
.apply-row .field input[type=number],
.apply-row .field select { flex:1; min-width:0; padding:8px 10px; border-radius:10px; border:1px solid rgba(255,255,255,0.15); background:rgba(255,255,255,0.06); color:#fff; }
.apply-actions { display:flex; align-items:center; justify-content:space-between; gap:10px; }
.apply-actions .left { display:flex; gap:10px; align-items:center; }
.apply-actions button { border:none; border-radius:12px; padding:10px 14px; cursor:pointer; background:linear-gradient(120deg,#5ad6ff,#8cefff); color:#041126; font-weight:700; }
.apply-actions .reset { background:transparent; color:#a5b4ce; border:1px solid rgba(255,255,255,0.2); }
.select-all { display:flex; gap:8px; align-items:center; }
.cam-list { display:grid; grid-template-columns: repeat(auto-fit,minmax(280px, 1fr)); gap:12px; }

.cam-card{
  border:1px solid rgba(255,255,255,0.12);
  border-radius:20px;
  padding:18px 20px;
  background:rgba(9,20,40,0.9);
  display:flex; flex-direction:column; gap:14px;
  box-shadow:0 18px 42px rgba(0,0,0,0.35);
}
.cam-card.checked{ border-color:#5ad6ff; box-shadow:0 22px 48px rgba(90,214,255,0.25); }
.card-select{ display:flex; justify-content:space-between; align-items:center; gap:12px; }
.card-select label{ display:flex; align-items:center; gap:8px; font-size:.9rem; color:#a5b4ce; }
.card-select input[type=checkbox]{ width:18px; height:18px; accent-color:#5ad6ff; }
.status-pill{ padding:6px 14px; border-radius:999px; font-size:.78rem; border:1px solid rgba(255,255,255,0.2); }
.status-pill.on{ background:rgba(46,204,113,0.15); color:#9af3c0; border-color:rgba(46,204,113,0.5); }
.status-pill.off{ background:rgba(231,76,60,0.18); color:#ffb0a4; border-color:rgba(231,76,60,0.5); }
.card-heading{ display:flex; justify-content:space-between; gap:16px; flex-wrap:wrap; }
.cam-name{ font-weight:700; font-size:1.05rem; }
.cam-id{ color:#a5b4ce; font-size:.82rem; }
.card-dates{ display:flex; flex-direction:column; gap:4px; font-size:.78rem; color:#a5b4ce; text-align:left; direction:ltr; }
.flash{ padding:10px 12px; border-radius:10px; font-size:.86rem; }
.flash.success{ background: rgba(46,204,113,0.12); color:#9af3c0; border:1px solid rgba(46,204,113,0.35); }
.flash.error{ background: rgba(231,76,60,0.12); color:#ffb0a4; border:1px solid rgba(231,76,60,0.35); }
.cam-inline-form{ display:flex; flex-direction:column; gap:12px; }
.form-grid{ display:grid; grid-template-columns: repeat(auto-fit,minmax(150px,1fr)); gap:12px 16px; }
.form-grid label{ display:flex; flex-direction:column; gap:6px; font-size:.86rem; }
.form-grid input, .form-grid select{ border-radius:12px; border:1px solid rgba(255,255,255,0.15); padding:10px 12px; background:rgba(255,255,255,0.06); color:#fff; }
.form-actions{ display:flex; justify-content:flex-end; gap:8px; }
.save-inline{ border:none; border-radius:14px; padding:10px 16px; font-weight:700; cursor:pointer; background:linear-gradient(120deg,#5ad6ff,#8cefff); color:#041126; }
.empty{ opacity:.85; padding:12px; border-radius:12px; border:1px dashed rgba(255,255,255,0.2); }
.header-bar{ display:flex; align-items:center; justify-content:space-between; padding:12px 0; }
.header-bar a{ color:#9ad8ff; }
</style>
</head>
<body class="gallery-bg">
<header class="app-header">
  <div class="header-wrap">
    <div class="app-logo">
      <span class="logo-spark">⚡</span>
      <div>
        <strong>تنظیمات دوربین‌ها</strong>
        <small>مدیریت و بروزرسانی تکی و گروهی</small>
      </div>
    </div>
    <div class="header-actions">
      <a href="gallery.php">گالری</a>
      <span style="opacity:.5">|</span>
      <a href="logout.php">خروج</a>
    </div>
  </div>
</header>

<main class="settings-page">
  <div class="panel-heading">
    <div>
      <h1>🎛️ مدیریت تنظیمات</h1>
      <small>مرتب‌سازی پیش‌فرض: بر اساس Device ID از کوچک به بزرگ</small>
    </div>
    <span class="count-chip"><?php echo cs_h((string)$resultCount); ?> دستگاه</span>
  </div>

  <?php if ($g = cs_flash_get('global')): ?>
    <div class="flash <?php echo $g['type'] === 'success' ? 'success' : 'error'; ?>">
      <?php echo cs_h($g['message']); ?>
    </div>
  <?php endif; ?>

  <form class="filters glass-sub" method="get" action="camera_settings.php">
    <div class="filter-item">
      <label for="q">جستجو (نام/ID)</label>
      <input type="text" id="q" name="q" value="<?php echo cs_h($q); ?>" placeholder="مثلاً CAM-12">
    </div>
    <div class="filter-item">
      <label for="frame">اندازه فریم</label>
      <select id="frame" name="frame">
        <option value="">همه اندازه‌ها</option>
        <?php foreach (ALLOWED_FRAMESIZES as $key => $label): ?>
          <option value="<?php echo cs_h($key); ?>" <?php echo ($frameFilter === $key ? 'selected' : ''); ?>>
            <?php echo cs_h($key . ' – ' . $label); ?>
          </option>
        <?php endforeach; ?>
      </select>
    </div>
    <div class="filter-item">
      <label for="capture">وضعیت کپچر</label>
      <select id="capture" name="capture">
        <option value="" <?php echo ($captureParam === '' ? 'selected' : ''); ?>>هر دو</option>
        <option value="1" <?php echo ($captureParam === '1' ? 'selected' : ''); ?>>فعال</option>
        <option value="0" <?php echo ($captureParam === '0' ? 'selected' : ''); ?>>غیرفعال</option>
      </select>
    </div>
    <div class="filter-item">
      <label for="sort">مرتب‌سازی</label>
      <select id="sort" name="sort">
        <?php foreach ($sortOptions as $k => $label): ?>
          <option value="<?php echo cs_h($k); ?>" <?php echo ($sortKey === $k ? 'selected' : ''); ?>>
            <?php echo cs_h($label); ?>
          </option>
        <?php endforeach; ?>
      </select>
    </div>
    <div class="filter-item" style="align-self:end;">
      <button type="submit" class="save-inline">اعمال فیلتر</button>
      <a class="apply-actions reset" href="camera_settings.php" style="margin-right:8px; padding:9px 12px; border-radius:12px; border:1px solid rgba(255,255,255,0.2);">ریست</a>
    </div>
  </form>

  <!-- پنل اعمال گروهی: یک فرم مستقل که با JS لیست دستگاه‌های انتخابی را دریافت می‌کند -->
  <form id="bulkForm" class="apply-board" method="post" action="<?php echo 'camera_settings.php' . cs_build_query($stateQuery); ?>">
    <input type="hidden" name="action" value="bulk_apply">
    <input type="hidden" name="bulk_devices" id="bulk_devices" value="">
    <div class="apply-row">
      <div class="field">
        <input type="checkbox" id="apply_interval" name="apply_interval">
        <label for="apply_interval">بازهٔ زمانی (دقیقه)</label>
        <input type="number" name="bulk_interval_minutes" min="1" max="240" placeholder="مثلاً 15">
      </div>
      <div class="field">
        <input type="checkbox" id="apply_quality" name="apply_quality">
        <label for="apply_quality">کیفیت JPEG</label>
        <input type="number" name="bulk_quality" min="5" max="63" placeholder="مثلاً 10">
      </div>
      <div class="field">
        <input type="checkbox" id="apply_frame_size" name="apply_frame_size">
        <label for="apply_frame_size">اندازهٔ فریم</label>
        <select name="bulk_frame_size">
          <?php foreach (ALLOWED_FRAMESIZES as $key => $label): ?>
            <option value="<?php echo cs_h($key); ?>"><?php echo cs_h($key . ' – ' . $label); ?></option>
          <?php endforeach; ?>
        </select>
      </div>
      <div class="field">
        <input type="checkbox" id="apply_capture" name="apply_capture">
        <label for="apply_capture">وضعیت کپچر</label>
        <select name="bulk_capture_enabled">
          <option value="1">فعال</option>
          <option value="0">غیرفعال</option>
        </select>
      </div>
    </div>
    <div class="apply-actions">
      <div class="left">
        <div class="select-all">
          <input type="checkbox" id="check_all">
          <label for="check_all">انتخاب همه</label>
        </div>
        <div id="selected_count" style="color:#a5b4ce;">۰ انتخاب</div>
      </div>
      <div>
        <button type="submit">اعمال گروهی</button>
      </div>
    </div>
  </form>

  <?php if (empty($items)): ?>
    <div class="empty">دوربینی با فیلترهای فعلی یافت نشد.</div>
  <?php else: ?>
    <section class="cam-list">
      <?php foreach ($items as $cam): ?>
        <?php $flash = cs_flash_get('cam-' . $cam['device_id']); ?>
        <div class="cam-card" id="cam-<?php echo cs_h($cam['device_id']); ?>">
          <div class="card-select">
            <!-- چک‌باکس انتخاب گروهی (خارج از فرم تکی) -->
            <label>
              <input type="checkbox" class="sel-cam" data-device="<?php echo cs_h($cam['device_id']); ?>">
              <span>انتخاب برای گروهی</span>
            </label>
            <span class="status-pill <?php echo $cam['capture_enabled'] ? 'on' : 'off'; ?>">
              <?php echo $cam['capture_enabled'] ? 'کپچر فعال' : 'کپچر غیرفعال'; ?>
            </span>
          </div>

          <div class="card-heading">
            <div>
              <div class="cam-name"><?php echo cs_h($cam['display_name']); ?></div>
              <small class="cam-id">ID: <?php echo cs_h($cam['device_id']); ?></small>
            </div>
            <div class="card-dates">
              <span>ایجاد: <?php echo cs_h(cs_format_dt($cam['created_at'])); ?></span>
              <span>آپدیت: <?php echo cs_h(cs_format_dt($cam['updated_at'])); ?></span>
            </div>
          </div>

          <?php if ($flash): ?>
            <div class="flash <?php echo $flash['type'] === 'success' ? 'success' : 'error'; ?>">
              <?php echo cs_h($flash['message']); ?>
            </div>
          <?php endif; ?>

          <!-- فرم ذخیرهٔ تکی -->
          <form class="cam-inline-form" method="post" action="<?php echo 'camera_settings.php' . cs_build_query($stateQuery) . '#cam-' . rawurlencode($cam['device_id']); ?>">
            <input type="hidden" name="action" value="save_single">
            <input type="hidden" name="device_id" value="<?php echo cs_h($cam['device_id']); ?>">
            <div class="form-grid">
              <label>
                <span>بازهٔ زمانی (دقیقه)</span>
                <input type="number" name="interval_minutes" min="1" max="240" step="1"
                       value="<?php echo cs_h((string)$cam['interval_minutes']); ?>">
              </label>
              <label>
                <span>کیفیت JPEG (۵ تا ۶۳)</span>
                <input type="number" name="quality" min="5" max="63" step="1"
                       value="<?php echo cs_h((string)$cam['quality']); ?>">
              </label>
              <label>
                <span>اندازهٔ فریم</span>
                <select name="frame_size">
                  <?php foreach (ALLOWED_FRAMESIZES as $key => $label): ?>
                    <option value="<?php echo cs_h($key); ?>" <?php echo (strtoupper($cam['frame_size']) === $key ? 'selected' : ''); ?>>
                      <?php echo cs_h($key . ' – ' . $label); ?>
                    </option>
                  <?php endforeach; ?>
                </select>
              </label>
              <label>
                <span>فعال‌سازی کپچر</span>
                <input type="checkbox" name="capture_enabled" <?php echo $cam['capture_enabled'] ? 'checked' : ''; ?>>
              </label>
            </div>
            <div class="form-actions">
              <button type="submit" class="save-inline">ذخیرهٔ همین کارت</button>
            </div>
          </form>
        </div>
      <?php endforeach; ?>
    </section>
  <?php endif; ?>
</main>

<script>
// انتخاب همه/بروزرسانی شمارنده و همگام‌سازی با فرم گروهی
(function(){
  const checkAll = document.getElementById('check_all');
  const selectedCount = document.getElementById('selected_count');
  const bulkForm = document.getElementById('bulkForm');
  const bulkDevicesInput = document.getElementById('bulk_devices');

  function getCardCheckboxes() {
    return Array.from(document.querySelectorAll('.sel-cam'));
  }
  function updateCardVisuals() {
    getCardCheckboxes().forEach(cb => {
      const card = cb.closest('.cam-card');
      if (!card) return;
      if (cb.checked) card.classList.add('checked');
      else card.classList.remove('checked');
    });
  }
  function updateSelectedCount() {
    const n = getCardCheckboxes().filter(cb => cb.checked).length;
    selectedCount.textContent = (n === 0) ? '۰ انتخاب' : (n + ' انتخاب');
  }
  function syncBulkDevices() {
    const ids = getCardCheckboxes().filter(cb => cb.checked).map(cb => cb.dataset.device);
    bulkDevicesInput.value = ids.join(',');
  }

  // رویداد انتخاب همه
  if (checkAll) {
    checkAll.addEventListener('change', function(){
      getCardCheckboxes().forEach(cb => cb.checked = checkAll.checked);
      updateCardVisuals();
      updateSelectedCount();
    });
  }

  // رویداد هر کارت
  document.addEventListener('change', function(e){
    if (e.target && e.target.classList.contains('sel-cam')) {
      updateCardVisuals();
      updateSelectedCount();
    }
  });

  // پیش از ارسال فرم گروهی، دستگاه‌ها را همگام کن
  if (bulkForm) {
    bulkForm.addEventListener('submit', function(e){
      syncBulkDevices();
      if (!bulkDevicesInput.value) {
        e.preventDefault();
        alert('هیچ دوربینی برای اعمال گروهی انتخاب نشده است.');
        return;
      }
      // دست‌کم یک فیلد تیک خورده؟
      const anyField =
        document.getElementById('apply_interval').checked ||
        document.getElementById('apply_quality').checked ||
        document.getElementById('apply_frame_size').checked ||
        document.getElementById('apply_capture').checked;
      if (!anyField) {
        e.preventDefault();
        alert('هیچ فیلدی برای اعمال گروهی انتخاب نشده است.');
      }
    });
  }

  // اگر پارامتر focus در URL بود، به کارت متناظر اسکرول نرم انجام بده
  const params = new URLSearchParams(window.location.search);
  const focus = params.get('focus');
  if (focus) {
    const el = document.getElementById('cam-' + focus);
    if (el) el.scrollIntoView({behavior:'smooth', block:'center'});
  }

  // شروع: شمارنده را به‌روز کن
  updateSelectedCount();
})();
</script>
</body>
</html>
