<?php
// camera_settings_model.php
// مدل مدیریت تنظیمات دوربین‌ها + اعتبارسنجی + لیست همه دوربین‌ها

if (!defined('DEFAULT_SETTINGS')) {
    define('DEFAULT_SETTINGS', array(
        'interval_minutes' => 60,
        'quality'          => 10,
        'frame_size'       => 'SVGA',
        'capture_enabled'  => true,
    ));
}

if (!defined('ALLOWED_FRAMESIZES')) {
    define('ALLOWED_FRAMESIZES', array(
        'VGA'  => '640x480',
        'SVGA' => '800x600',
        'XGA'  => '1024x768',
        'SXGA' => '1280x1024',
        'UXGA' => '1600x1200',
    ));
}

// Converts legacy keys or literal resolutions (e.g. 1024x768) to a canonical frame_size key.
function normalize_frame_size($value)
{
    if ($value === null) {
        return null;
    }

    $value = trim((string)$value);
    if ($value === '') {
        return null;
    }

    $upper = strtoupper($value);
    if (isset(ALLOWED_FRAMESIZES[$upper])) {
        return $upper;
    }

    $resolution = strtoupper(str_replace(array('×', '*'), 'X', $value));
    $resolution = preg_replace('/[^0-9X]/', '', $resolution);
    $map = array(
        '640X480'   => 'VGA',
        '800X600'   => 'SVGA',
        '1024X768'  => 'XGA',
        '1280X1024' => 'SXGA',
        '1600X1200' => 'UXGA',
    );

    return ($resolution !== '' && isset($map[$resolution])) ? $map[$resolution] : null;
}

// Converts truthy/falsy inputs to bool; returns null for invalid values.
function convert_to_bool_value($value)
{
    if (is_bool($value)) {
        return $value;
    }

    if (is_string($value)) {
        $normalized = strtolower(trim($value));
        if ($normalized === '1' || $normalized === 'true' || $normalized === 'on' || $normalized === 'yes') {
            return true;
        }
        if ($normalized === '0' || $normalized === 'false' || $normalized === 'off' || $normalized === 'no') {
            return false;
        }
        return null;
    }

    if (is_int($value) || is_float($value)) {
        if ((int)$value === 1) {
            return true;
        }
        if ((int)$value === 0) {
            return false;
        }
        return null;
    }

    return null;
}

// Applies defaults, normalizes legacy keys, and keeps capture_enabled consistent.
function merge_settings_with_defaults($stored)
{
    if (!is_array($stored)) {
        $stored = array();
    }

    if (isset($stored['framesize']) && !isset($stored['frame_size'])) {
        $stored['frame_size'] = $stored['framesize'];
    }

    $merged = array_merge(DEFAULT_SETTINGS, $stored);

    $frameKey = normalize_frame_size(isset($merged['frame_size']) ? $merged['frame_size'] : null);
    if ($frameKey === null) {
        $frameKey = DEFAULT_SETTINGS['frame_size'];
    }
    $merged['frame_size'] = $frameKey;
    unset($merged['framesize']);

    $captureValue = convert_to_bool_value(isset($merged['capture_enabled']) ? $merged['capture_enabled'] : DEFAULT_SETTINGS['capture_enabled']);
    if ($captureValue === null) {
        $captureValue = DEFAULT_SETTINGS['capture_enabled'];
    }
    $merged['capture_enabled'] = $captureValue;

    $merged['interval_minutes'] = (int)$merged['interval_minutes'];
    $merged['quality'] = (int)$merged['quality'];

    return $merged;
}

function getAllCameras($pdo)
{
    $sql = "SELECT device_id, name, settings, created_at, updated_at
            FROM cameras
            ORDER BY (name IS NULL) ASC, name ASC, device_id ASC";
    $stmt = $pdo->query($sql);
    if (!$stmt) {
        return array();
    }

    $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);
    $result = array();

    foreach ($rows as $row) {
        $stored = array();
        if (!empty($row['settings'])) {
            $decoded = json_decode($row['settings'], true);
            if (is_array($decoded)) {
                $stored = $decoded;
            }
        }

        $result[] = array(
            'device_id'  => $row['device_id'],
            'name'       => isset($row['name']) ? $row['name'] : null,
            'settings'   => merge_settings_with_defaults($stored),
            'created_at' => isset($row['created_at']) ? $row['created_at'] : null,
            'updated_at' => isset($row['updated_at']) ? $row['updated_at'] : null,
        );
    }

    return $result;
}

function getCameraSettings($pdo, $deviceId)
{
    $stmt = $pdo->prepare("SELECT settings FROM cameras WHERE device_id = :id LIMIT 1");
    $stmt->execute(array(':id' => $deviceId));
    $row = $stmt->fetch(PDO::FETCH_ASSOC);
    if (!$row) {
        return null;
    }

    $stored = array();
    if (!empty($row['settings'])) {
        $decoded = json_decode($row['settings'], true);
        if (is_array($decoded)) {
            $stored = $decoded;
        }
    }

    return merge_settings_with_defaults($stored);
}

function validateCameraSettings(&$input)
{
    $errors = array();

    if (!isset($input['interval_minutes']) || $input['interval_minutes'] === '') {
        $errors['interval_minutes'] = 'بازه زمانی معتبر نیست.';
    } else {
        $val = (int)$input['interval_minutes'];
        if ($val < 1 || $val > 240) {
            $errors['interval_minutes'] = 'بازه زمانی باید بین ۱ تا ۲۴۰ دقیقه باشد.';
        } else {
            $input['interval_minutes'] = $val;
        }
    }

    if (!isset($input['quality']) || $input['quality'] === '') {
        $errors['quality'] = 'کیفیت معتبر نیست.';
    } else {
        $q = (int)$input['quality'];
        if ($q < 5 || $q > 63) {
            $errors['quality'] = 'کیفیت باید بین ۵ تا ۶۳ باشد.';
        } else {
            $input['quality'] = $q;
        }
    }

    if (!isset($input['frame_size']) && isset($input['framesize'])) {
        $input['frame_size'] = $input['framesize'];
    }

    $frameKey = normalize_frame_size(isset($input['frame_size']) ? $input['frame_size'] : null);
    if ($frameKey === null) {
        $errors['frame_size'] = 'اندازه فریم انتخاب‌شده مجاز نیست.';
    } else {
        $input['frame_size'] = $frameKey;
    }
    unset($input['framesize']);

    if (array_key_exists('capture_enabled', $input)) {
        $capture = convert_to_bool_value($input['capture_enabled']);
        if ($capture === null) {
            $errors['capture_enabled'] = 'مقدار فعال‌سازی کپچر معتبر نیست.';
        } else {
            $input['capture_enabled'] = $capture;
        }
    } else {
        $input['capture_enabled'] = DEFAULT_SETTINGS['capture_enabled'];
    }

    return $errors;
}

function saveCameraSettings($pdo, $deviceId, $input)
{
    $normalized = array(
        'interval_minutes' => isset($input['interval_minutes']) ? $input['interval_minutes'] : null,
        'quality'          => isset($input['quality']) ? $input['quality'] : null,
        'frame_size'       => isset($input['frame_size']) ? $input['frame_size'] : null,
        'capture_enabled'  => array_key_exists('capture_enabled', $input) ? $input['capture_enabled'] : null,
    );

    if ($normalized['frame_size'] === null && isset($input['framesize'])) {
        $normalized['frame_size'] = $input['framesize'];
    }

    $errors = validateCameraSettings($normalized);
    if (!empty($errors)) {
        return array('ok' => false, 'errors' => $errors);
    }

    $stmtSel = $pdo->prepare("SELECT settings FROM cameras WHERE	device_id = :id LIMIT 1");
    $stmtSel->execute(array(':id' => $deviceId));
    $row = $stmtSel->fetch(PDO::FETCH_ASSOC);

    $finalSettings = DEFAULT_SETTINGS;
    if ($row && !empty($row['settings'])) {
        $prev = json_decode($row['settings'], true);
        if (is_array($prev)) {
            $finalSettings = merge_settings_with_defaults($prev);
        }
    }

    $finalSettings['interval_minutes'] = $normalized['interval_minutes'];
    $finalSettings['quality']          = $normalized['quality'];
    $finalSettings['frame_size']       = $normalized['frame_size'];
    $finalSettings['capture_enabled']  = $normalized['capture_enabled'];
    unset($finalSettings['framesize']);

    $json = json_encode($finalSettings, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);

    $sql = "INSERT INTO cameras (device_id, settings)
            VALUES (:id, :settings)
            ON DUPLICATE KEY UPDATE settings = VALUES(settings), updated_at = CURRENT_TIMESTAMP";
    $stmt = $pdo->prepare($sql);
    $stmt->execute(array(
        ':id'       => $deviceId,
        ':settings' => $json
    ));

    return array('ok' => true, 'errors' => array());
}
